<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Se incluye la conexión (ya configurada con PDO)
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

try {
    // Obtener y decodificar el cuerpo de la solicitud
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    // Validar datos de entrada
    if (!isset($datos[0], $datos[1], $datos[2]) || 
        !is_numeric($datos[0]) || !is_numeric($datos[1]) || !is_numeric($datos[2])) {
        error_response("Parámetros inválidos.");
    }

    // Preparar la consulta para ejecutar la función almacenada
    $sql = "SELECT * FROM listar_equivalencias(:producto_id, :empresa_id, :sucursal_id)";
    $stmt = $conn->prepare($sql);
    
    // Asignar parámetros de manera segura
    $stmt->bindValue(':producto_id', (int)$datos[0], PDO::PARAM_INT);
    $stmt->bindValue(':empresa_id', (int)$datos[1], PDO::PARAM_INT);
    $stmt->bindValue(':sucursal_id', (int)$datos[2], PDO::PARAM_INT);

    // Ejecutar la consulta
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Verificar si hay datos
    if (!$rows) {
        error_log("⚠️ No se encontraron equivalencias con esos filtros.");
        $rows = [];
    }

    // Cerrar la conexión y liberar recursos
    $stmt = null;
    $conn = null;

    // Enviar respuesta JSON
    print_json($rows);

} catch (PDOException $e) {
    error_log("❌ Error en la base de datos: " . $e->getMessage());
    error_response("Error interno en la base de datos.");
} catch (Exception $e) {
    error_log("❌ Error general: " . $e->getMessage());
    error_response("Error interno del servidor.");
}

// Función para imprimir JSON con validación de errores
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown JSON error');
            break;
    }
}

// Función para manejar errores y responder en formato JSON
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>
